include bt1ttl.inc



;
;
;	network-specific routines for booting
;
;

name	network


public	nt_reset;		resets network hardware and software
public	nt_cup;			tests for network control unit (board) present
public	nt_dvcrdy;		checks for device ready (server can load us)
public	nt_online;		put server on-line (ready to load)
public	nt_read;		read boot records
public	nt_quiesce;		quiesce (terminate) the network load




data	segment	public 'data';

include	bt1lrb.str
	extrn	lrb:byte;		load request block structure
include	bt1ntctl.str
	extrn	netctl:byte;		network variables and status
	extrn	ccb:byte;		omninet command control block
	extrn	send_buffer:byte;	transmission buffer
	extrn	receive_buffer:byte;	receive buffer for header
include	bt1ntlbl.str
	extrn	bootr:byte;		network label buffer

data	ends;




;
;
;	network board hardware values
;
;
omniadr		equ	0E810h;		address of Omninet board I/O
xrdybit		equ	08h;		transporter ready bit

echo		equ	002h;		omninet command . . . echo test
abort		equ	010h;		                      receive abort
init		equ	020h;		                      initialize board
transmit	equ	040h;		                      transmit
receive		equ	0F0h;		                      receive

;
;	define omninet card's i/o registers
;
ioports		segment		AT(omniadr)

		org	0
readnext_xptr	label	byte;	read at transporter with auto increment
wrram		label	byte;	write at transporter with auto increment
		org	1
readbyte_xptr	label	byte;	read at transporter
wrstrobe	label	byte;	command address (high,middle,low) & strobe
		org	2
rdlocounter	label	byte;	low byte of transporter
wrlocounter	label	byte;	low byte of transporter
		org	3
rdhicounter	label	byte;	high nibble of transporter
wrhicounter	label	byte;	high nibble of transporter
		org	7
rdstatus	label	byte;	bit3=transporter ready (08h)


ioports		ends




page
code	segment	public	'code'

extrn	time:near;		routine to wait 100n microseconds

cgroup group code;
dgroup group data;
assume cs:cgroup,ds:dgroup,es:ioports;

;
;	network error recovery retry counters
;
nrsp_to		equ	100;	timeout on receiving a message (10 seconds)
;				note, nrsp_to uses retry_pause for each trial
retry_count	equ	1000;	retry transmit count
send_msg_to	equ	1000;	gross timeout on send msg command (100ms)
recv_prep_to	equ	1000;	timeout on board's receive preparation (100ms)
retry_pause	equ	1000;	pause on send before retry (100ms)


;
;	error statuses generated by this module
;
xmto		equ	040h;	transmit timeout
unrc		equ	041h;	"recoverable" error was not recoverable
rsto		equ	042h;	board's receive set-up took too long
nrsp		equ	043h;	no response from the server
xsvr		equ	044h;	received message from wrong server
xseq		equ	045h;	received message with a bad sequence number
xopc		equ	046h;	received message with wrong operation code
xfnc		equ	047h;	server does not support the boot function


;
;	miscellaneous equates
;
boot_version	equ	1;	version of boot rom to tell server
last_server	equ	9;	servers 0 to 9, stations 10 to 63


page
;
;
;	high-level routines called by the control unit operation dispatcher
;
;



;
;	initialize the network variables
;
;	inputs:		none
;
;	outputs:	all registers destroyed
;
nt_reset	proc;

	mov	netctl.msg_seq_nr,0;	initialize the sequence number

	ret;

nt_reset	endp;




;
;	check for control unit (network transporter board) present
;
;	inputs:		none
;
;	outputs:	all registers destroyed
;			z-flag is nz if board is not there
;
nt_cup		proc;

	mov	ax,seg ioports;		address the I/O registers
	mov	es,ax;

	xor	bx,bx;			first pattern

test_next:;
	mov	dx,cs:tests[bx];	fetch the particular test pattern
	cmp	dx,0FFFFh;		done ?
	jz	is_there;		yes, omninet board is there

	call	set_pointer;		set omninet pointer to pattern

	mov	al,rdlocounter;		get least significant byte of pointer
	mov	ah,rdhicounter;		and high order byte of pointer
	and	ah,0Fh;			mask unused bits

	cmp	ax,dx;			read value matches ?
	jnz	not_there;		no

	inc	bx;
	inc	bx;			next test pattern
	jmp	test_next;

;
;	initialize the transporter board
;
is_there:;
	mov	ccb.command,init;	initialize command

	xor	dx,dx;
	mov	ccb.rra,dl;		high order byte is zero
	mov	ccb.rra+1,dl;		so is middle byte
	mov	ccb.rra+2,80h;		least significant byte of address

	call	set_80h;		do it

init_wait:;
	mov	ah,readbyte_xptr;
	cmp	ah,0ffh;		ready ?
	jz	init_wait;		no, wait

	mov	netctl.sta_id,ah;	save our address

	xor	ax,ax;			flag successful
not_there:;
	ret;

;
;	test patterns for the omninet board
;
tests	dw	0000h,0FFFh,0555h,0AAAh,0FFFFh;

nt_cup		endp;



;
;	test for device ready (server ready to load us)
;
;	inputs:		load request block set up
;
;	outputs:	load request block filled in
;			z-flag is nz if no server is ready
;			all registers destroyed
;
nt_dvcrdy	proc;

	mov	ax,seg ioports;		address the I/O registers
	mov	es,ax;

	xor	bl,bl;			start at server 0

;
;	do an echo test to see if a server's board is responding
;
check_this_one:;
	mov	ccb.command,echo;	command is echo test

	xor	dx,dx;
	mov	ccb.rra,dl;		high address is zero
	mov	ccb.rra+1,dl;		middle byte is zero
	mov	ccb.rra+2,80h;		start at 80h in omninet ram

	mov	ccb.socket,bl;		server to test for

	call	set_80h;		set up the command

command_accept_wait:;
	mov	ax,1;			wait 100 microseconds
	call	time;

	mov	ah,readbyte_xptr;	read the board
	cmp	ah,0FFh;		done ?
	jz	command_accept_wait;	not yet

	mov	netctl.ret_code,ah;	get omninet return status

	cmp	ah,0C0h;		board there ?
	jnz	try_next_server;	no, try next server

;
;	this server is there, send the "are you ready" message
;
	mov	send_buffer.opcode,devc_ready;		operation ready check

	mov	ax,net_header_size+send_devc_ready_size;
	push	ax;			size to send

	xor	ax,ax;			no buffer needed for receive data
	push	ax;
	push	ax;

	mov	ax,net_header_size+recv_devc_ready_size;
	push	ax;			expected size to receive

	call	do_trans;		perform the send and receive
	jnz	try_next_server;	the transmission failed

	mov	al,netctl.server_status;
	cmp	al,wait;		server wacked us ?
	jz	wacked;			yes, try it again

	or	al,al;			successful response ?
	jnz	try_next_server;	no, try another

	mov	ax,receive_buffer.rcvhand;
	mov	netctl.my_handle,ax;	save handle to use for server access

	mov	ax,lrb.dun;		get device and unit field
	and	al,0F0h;		clear unit number
	or	al,bl;			add in this server's number
	mov	lrb.dun,ax;		and save for later

	xor	ax,ax;			flag success in z-flag
	ret;

;
;	got a wack from the server, retry it
;
wacked:;
	mov	ax,retry_pause;		wait before retrying
	call	time;
	jmp	check_this_one;

;
;	this server failed, try the next on
;
try_next_server:;
	inc	bl;			up the server's number
	cmp	bl,last_server;		tried all of them ?
	ja	failed;			yes, error, no servers
	jmp	check_this_one;		no, try this one.

failed:;
	or	bl,bl;			set z-flag nz
	ret;				indicating failure in network boot

nt_dvcrdy	endp;



;
;	put device on-line (prepare to load)
;
;	inputs:		load request block set up
;
;	outputs:	load request block updated (status is error information)
;			z-flag is nz if an error occurred
;			all registers destroyed
;
nt_online	proc;

	mov	ax,seg ioports;		address the I/O registers
	mov	es,ax;

retry:;
	mov	bx,lrb.dun;		get unit number (server)
	and	bl,0Fh;

	mov	send_buffer.opcode,on_line;	request an on-line message

	mov	ax,net_header_size+send_on_line_size;
	push	ax;			size of data to send

	push	ds;			received data to label buffer
	lea	ax,ds:bootr;		(in data segment)
	push	ax;

	mov	ax,net_header_size+recv_on_line_size;
	push	ax;			size of data to receive

	call	do_trans;		perform the send and receive
	jnz	on_line_error;		the transmission failed

	mov	al,netctl.server_status;get server's response status
	cmp	al,wait;		server wacked us ?
	jz	wacked2;		yes, retry the operation

	or	al,al;			any error ?
	jnz	return_error;		yes.

;
;	fill in the lrb from the label
;
	mov	ax,bootr.blk_siz;
	mov	lrb.ssz,ax;		size of a block

	mov	ax,bootr.blk_nbr;
	mov	word ptr lrb.da,ax;	starting block number

	mov	ax,bootr.load;		load address
	mov	lrb.loadaddr,ax;

	mov	ax,bootr.paras;
	mov	lrb.loadpara,ax;	number of paragraphs in image

	mov	ax,word ptr bootr.start;
	mov	lrb.loadentry,ax;	entry address (offset)
	mov	ax,word ptr bootr.start+2;
	mov	lrb.loadentry+2,ax;	and segment

	mov	ax,bootr.max_cnt;
	mov	lrb.da+2,ax;		maximum blocks allowable

	xor	ax,ax;			flag no error
	ret;

;
;	server wacked us, wait and retry
;
wacked2:;
	mov	ax,retry_pause;
	call	time;
	jmp	retry;

;
;	transmission error in putting server on-line
;
on_line_error:;
	mov	al,netctl.ret_code;	load transmission's status

;
;	error occurred in putting device on-line
;
return_error:;
	mov	lrb.status,al;		pass along server's error status

	or	al,al;			flag an error
	ret;

nt_online	endp;



;
;	read a boot record from the server
;
;	inputs:		load request block set up
;
;	outputs:	load request block updated (status is error information)
;			z-flag is nz if an error occurred
;			all registers destroyed
;
nt_read		proc;

	mov	ax,seg ioports;		address the I/O registers
	mov	es,ax;

retry2:;
	mov	bx,lrb.dun;		get unit number (server)
	and	bl,0Fh;

	mov	send_buffer.opcode,read;	request a read

	mov	ax,lrb.da;		get block number to request
	mov	send_buffer.blk_nbr,ax;

	mov	ax,lrb.blkcnt;		get number of blocks required to read
	cmp	ax,lrb.da+2;		compare to max # can request
	jbe	no_leftovers;		can fill the whole request

	mov	ax,lrb.da+2;		can ask only for maximum allowed

no_leftovers:;
	mov	send_buffer.blk_cnt,ax;	request the maximum # blocks

	mov	dx,net_header_size+send_read_size;
	push	dx;			transmit header and 6 data bytes

	mov	dx,lrb.dma+2;		get address to load the image at
	push	dx;			segment
	mov	dx,lrb.dma;		and offset
	push	dx;

	mul	lrb.ssz;		# blocks times size of a block
	add	ax,net_header_size+recv_read_size;	header and ERROR byte
	push	ax;			yields maximum # bytes to receive

	call	do_trans;		perform the send and receive
	jnz	read_error;		the transmission failed

	mov	al,netctl.server_status;get server's response status
	cmp	al,wait;		server wacked us ?
	jz	wacked3;		yes, retry the operation

	or	al,al;			any error from server ?
	jnz	return_error2;		yes.

;
;	successfully read a portion of the load image, see if we're done
;
	mov	ax,send_buffer.blk_cnt;	get number of blocks read
	cmp	ax,lrb.blkcnt;		compare to requested # of blocks
	jz	finished;		read whole thing, done

	sub	lrb.blkcnt,ax;		decrement # blocks left to read

	add	lrb.da,ax;		increment block # to request

	mul	lrb.ssz;		number blocks read times size of a block
	mov	cl,4;
	shr	ax,cl;			(NOTE:  assumes block is multiple of 16)
	add	lrb.dma+2,ax;		increment dma address (segment)

	jmp	retry2;			read another portion of the image

;
;	server wacked us, retry the operation after waiting
;
wacked3:;
	mov	ax,retry_pause;
	call	time;
	jmp	retry2;


;
;	transmission error
;
read_error:;
	mov	al,netctl.ret_code;	get transmission error code

return_error2:;
	mov	lrb.status,al;		save status indication

	or	al,al;			flag an error occurred

finished:;
	ret;

nt_read		endp;



;
;	terminate the load process
;
;	inputs:		load request block
;
;	outputs:	all registers destroyed
;			z-flag is nz if an error occurred
;
nt_quiesce	proc;

	mov	ax,seg ioports;		address the I/O registers
	mov	es,ax;

	mov	bx,lrb.dun;		get unit number (server)
	and	bl,0Fh;

retry3:;
	mov	send_buffer.opcode,quiesce;	request quiesce

	mov	ax,net_header_size+send_quiesce_size;	send message size
	push	ax;

	push	ds;			received data to label buffer
	lea	ax,ds:bootr;		(in data segment)
	push	ax;

	mov	ax,net_header_size+recv_quiesce_size;	header, ERROR byte
	push	ax;

	call	do_trans;		perform the send and receive
	jnz	quiesce_error;		the transmission failed

	mov	al,netctl.server_status;get server's response status
	cmp	al,wait;		server wacked us ?
	jz	wacked4;		yes, retry the operation

	or	al,al;			any error ?

quiesce_error:
	ret;				return error indication

;
;	server wacked us, wait and retry the operation
;
wacked4:;
	mov	ax,retry_pause;
	call	time;
	jmp	retry3;

nt_quiesce	endp;



page


;
;
;	message-level transmission routines for the network
;
; 



;
;	perform a network transaction (send and receive)
;
;	inputs:		bl = server number
;
;			stack parameter 1 = transmitted message's size
;			stack parameter 2 = receive buffer's address (seg,offs)
;				will contain all data following ERROR byte
;			stack parameter 3 = return message's maximum size
;
;			all sizes must include 14 byte header (+1, on receive,
;			for the ERROR byte)
;
;			data portion of send_buffer must be set up
;
;
;	outputs:	all registers destroyed
;			net control structure updated
;			z-flag is nz if an error occurred
;



do_trans	proc;

	mov	bp,sp;			save base of our parameters

	mov	ax,seg ioports;		address the I/O registers
	mov	es,ax;
	assume	es:ioports;

	xor	ax,ax;
	mov	netctl.server_status,ah;
	mov	netctl.ret_code,ah;	initialize return statuses

;
;	send the message to the server
;

;
;	put on the routing headers for the transmitted message
;
	mov	send_buffer.msghdr_size,net_header_size;	size of header

	mov	send_buffer.msghdr_fmtid,protocol_version;

	xor	ax,ax;				no segmentation information
	mov	word ptr send_buffer.msghdr_ltime,ax;
	mov	send_buffer.msghdr_segoff,ax;	(and no offset address)

	mov	ah,address_size;
	mov	send_buffer.msghdr_dadln,ah;	destination address's length
	mov	send_buffer.msghdr_sadln,ah;	source address, too

	mov	al,bl;				server's number
	mov	ah,(fsocket-80h)*4+boot_function;	socket and type
	mov	send_buffer.msghdr_daddr,ax;	(socket is encoded in 2 msbits)

	mov	al,netctl.sta_id;	get this station's number
	mov	ah,(ssocket-80h)*4+boot_ret;	socket and returned message's type
	mov	send_buffer.msghdr_saddr,ax;	(socket is encoded in 2 msbits)

	inc	netctl.msg_seq_nr;	sequence number
	mov	ax,netctl.msg_seq_nr;	get the message's sequence number
	mov	send_buffer.msghdr_duid,ax;

	mov	ax,netctl.my_handle;	set handle to use for server access
	mov	send_buffer.handle,ax;

	mov	send_buffer.version,boot_version;	version of boot rom

;
;	now issue the transmission
;
	mov	ah,transmit;		transmit command
	mov	cx,[bp+8];		length to send
	call	ccb_setup;		set up the command control block

	mov	dx,100h;		location in omninet ram to write at
	lea	si,dgroup:send_buffer;	where to get data from
	mov	cx,[bp+8];		size of the request
	call	load_msg;		copy transmit message to omninet ram

;
;	perform the I/O and retry it
;
	mov	cx,retry_count;		count of allowable retries

retry_error:;
	call	set_80h;		issue ccb's command

	mov	di,send_msg_to;		gross transmission timeout

;
;	wait for message transmission - no completion is disaster
;
wait_for_read:;
	mov	ah,readbyte_xptr;	check for completion
	cmp	ah,0FFh;		done ?
	jnz	got_it;			yes.

	mov	ax,1;			wait 100 microseconds
	call	time;

	dec	di;			decrement gross timeout counter
	jnz	wait_for_read;		no, keep waiting

;
;	gross timeout
;
	mov	netctl.ret_code,xmto;	error, to long to transmit
	jmp	err_exit;

;
;	has been transmitted
;
got_it:;

;
;	return if transporter error not retry type
;
	test	ah,80h;			error ?
	jz	sent;			no, return okay

	cmp	ah,80h;			retryable error ?
	jz	retry9;			yes.
	cmp	ah,82h;
	jz	retry9;

	mov	netctl.ret_code,ah;	read the status from result vector
	jmp	err_exit;		no, bad error.
;
;	retry a recoverable error
;
retry9:;
	mov	ax,retry_pause;		wait before retrying
	call	time;
	loop	retry_error;

	mov	netctl.ret_code,unrc;	couldn't recover from the error
	jmp	err_exit;		too many retries
 
;
;	the transmission was successfully completed, now receive
;
sent:;

;
;	ccb setup for receive
;

	mov	ah,receive;		receive command
	mov	cx,[bp+2];		maximum receive size
	call	ccb_setup;

	call	set_80h;		issue command to transporter

;
;	timeout how long the board takes to go into receive mode
;

	mov	dx,recv_prep_to;	timeout for board to accept receive

timeout_board_setup:;
	mov	al,readbyte_xptr;
	cmp	al,0FFh;
	jnz	in_receive;		board is in receive mode

	mov	ax,1;
	call	time;			wait 100 microseconds

	dec	dx;			timeout counter
	jnz	timeout_board_setup;

;
;	board timed out, non-operational
;
	call	abort_recv;		abort the receive
	mov	netctl.ret_code,rsto;	receive set-up timeout
	jmp	err_exit;		and quit

;
;	board ready to receive
;
in_receive:;
	cmp	al,0FEh;		everything okay ?
	jz	recv_wait;		yes
	mov	netctl.ret_code,al;	store the status
	jmp	err_exit;		no, error

;
;	wait to receive a message
;
recv_wait:;
	mov	dx,nrsp_to;		set gross timeout for non-response
response_loop:;
	mov	ah,readbyte_xptr;
	cmp	ah,0FEh;		received stuff yet ?
	jnz	have_data;		yes, look at it.

	mov	ax,retry_pause;		wait until retrying
	call	time;

	dec	dx;
	jnz	response_loop;		decrement retry loop

;
;	server is non-responsive
;
	call	abort_recv;		abort the receive
	mov	netctl.ret_code,nrsp;	no response error code
	jmp	err_exit;		error - timeout on response

;
;	do if good receipt of message, get server number and length
;
have_data:;
	or	ah,ah;
	jz	good_status;
	mov	netctl.ret_code,ah;	save status returned
	jmp	err_exit;

good_status:;
	mov	ah,readnext_xptr;	update ram pointer to source id

	mov	ah,readnext_xptr;	get source id of message
	mov	netctl.src_id,ah;	store server's number

	mov	ah,readnext_xptr;	high order byte of message length
	mov	al,readnext_xptr;	low order byte of message length
	mov	netctl.mlen,ax;

;
;	store received message into buffers
;
	mov	dx,100h;		address of received data
	call	set_pointer;

;
;	first store the header and error flag in receive_buffer
;
	mov	cx,net_header_size+recv_read_size;	header plus error byte
	lea	si,ds:receive_buffer;

readhdr_loop:;
	mov	ah,readnext_xptr;	read omninet ram
	mov	ds:[si],ah;		store in receive_buffer
	inc	si;			next buffer location
	loop	readhdr_loop;		until finished

;
;	next store any data (label or load image) into caller's buffer
;
	mov	cx,netctl.mlen;		receive size
	sub	cx,net_header_size+recv_read_size;	minus header and error
	jz	no_data;		message is just header and error byte

	mov	si,[bp+4];		receive buffer
	push	ds;			save ds
	mov	ds,[bp+6];		receive buffer segment

readdata_loop:;
	mov	ah,readnext_xptr;	read omninet ram
	mov	ds:[si],ah;		store in caller's area
	inc	si;			increment result buffer offset
	loop	readdata_loop;		until count (cx) exhausted

	pop	ds;			restore ds

no_data:;
	mov	ah,byte ptr receive_buffer.rcvhdr_daddr+1;
	and	ah,03Fh;		only function bits (remove socket)
	cmp	ah,boot_ret;		received correct type of message ?
	jz	right_function;		yes, skip

;
;	wrong function type returned, error
;
	mov	netctl.ret_code,xopc;	preset to wrong opcode in message

right_function:;
	mov	ah,receive_buffer.rcvhdr_ipdu;	get error indication
	or	ah,ah;			any error ?
	jz	supported_function;	no, function was accepted

;
;	error, server doesn't support the requested function
;
	mov	netctl.ret_code,xfnc;	error, function not supported

supported_function:;
	mov	ax,receive_buffer.rcvhdr_duid;
	mov	netctl.rcv_seq_nr,ax;	save message number
	cmp	ax,netctl.msg_seq_nr;	right sequence number ?
	jz	right_sequence;		yes, skip

;
;	wrong sequence number, error
;
	mov	netctl.ret_code,xseq;

right_sequence:;
	mov	ah,byte ptr receive_buffer.rcvhdr_saddr;
	cmp	ah,bl;			correct server ?
	jz	right_server;		yes, skip

;
;	wrong server responded, error
;
	mov	netctl.ret_code,xsvr;	error, wrong server

right_server:;
	mov	ah,receive_buffer.err;	get server's error status
	mov	netctl.server_status,ah;

	mov	ah,netctl.ret_code;	get return status
	or	ah,ah;			flag if errors
	ret	8;

;
;	error return, something's wrong
;
err_exit:;

	mov	ah,1;			flag an error in z-flag
	or	ah,ah;
	ret	8;

do_trans	endp;



;
;	abort a receive
;
;	inputs:		none
;
;	outputs:	ax, dx, di, si destroyed
;
abort_recv	proc;

	mov	ah,abort;		set command to end receive
	mov	ccb.command,ah;

	mov	dx,10h;			ccb's address
	mov	di,90h;			result vector's address
	call	set_command;

	mov	dx,recv_prep_to;	timeout for board's receive preparation
abort_wait:;
	mov	ah,readbyte_xptr;
	cmp	ah,0FFH;
	jnz	aborted;		board has accepted the abort

	mov	ax,1;			otherwise wait before retrying
	call	time;			for 100 microseconds

	dec	dx;			retry count
	jnz	abort_wait;		if too long, assume aborted

aborted:;
	ret;

abort_recv	endp;





;
;	set up command control block for omninet board, receive or send
;
;	inputs:		ah - command
;			bl - server to communicate with
;			cx - transmit or maximum receive size
;
;	outputs:	cx destroyed
;
ccb_setup	proc;

	mov	ccb.command,ah;		set up the command to issue

	mov	byte ptr ccb.len,ch;	size to transmit/receive
	mov	byte ptr ccb.len+1,cl;

	mov	ccb.socket,fsocket;	socket used to transmit to
	cmp	ah,transmit;		sending ?
	jz	sending;		yes, skip
	mov	ccb.socket,ssocket;	receiving socket number

sending:;
	xor	cx,cx;
	mov	byte ptr ccb.rra,cl;	msb of result record
	mov	byte ptr ccb.rra+1,cl;	result record at 080h
	mov	ch,080h;
	mov	byte ptr ccb.rra+2,ch;	lsb of result record


	mov	byte ptr ccb.daddr,cl;	data address is 100h
	mov	ch,01h;
	mov	byte ptr ccb.daddr+1,ch;
	mov	byte ptr ccb.daddr+2,cl;
	
	mov	ccb.control,cl;		no control information (length of 0)

	mov	ccb.dest,bl;		server to access

	ret;

ccb_setup	endp;





;
;	copy data to omninet ram at location from msg_ptr for msg_size
;
;	inputs:		dx = location to load message at
;			ds:si = message pointer
;			cx = message size
;
;	outputs:	ax, cx, si destroyed
;

load_msg	proc;

	call	set_pointer;		set address in Omninet ram

;
;	write the data to the board
;

writedata_loop:;
	mov	ah,ds:[si];		get data from caller's buffer
	mov	wrram,ah;		write to OMNINET board
	inc	si;			update buffer pointer
	loop	writedata_loop;		continue util count (cx) exhausted

	ret;

load_msg	endp;
    


;
;	set_command with result vector at 80h and ccb at 00h
;	(see below for details)
;
set_80h		proc;

	xor	dx,dx;		ccb at 0 in omninet ram
	mov	di,80h;		result vector's address in omninet ram
;				fall through to set_command

set_80h		endp;



;
;	load ccb into omninet ram, set the result record
;	initially to 0FF hex, issue the command, and
;	leave omninet ram pointer at result buffer.
;
;	inputs:		dx = location to load ccb at
;			di = result omninet ram address
;			
;	outputs:	dx, di, si destroyed
;

set_command	proc;

	push	ax;
	push	cx;			save caller's registers

 	mov	si,offset ccb;		address of the command control block
	mov	cx,size ccbs;		size of the control block
	call	load_msg;		copy ccb to board

	xchg	dx,di;			get result address
	call	set_pointer;		set pointer to result address

	mov	wrram,0FFh;		tell board to get ready

;
;	strobe the command into the hardware
;
	xchg	dx,di;			get ccb's address in dx
ready_wait:;
	test	rdstatus,xrdybit;	ready ?
	jz	ready_wait;		loop until ready

	mov	wrstrobe,0;		high byte of address = 0, only 4k of ram

read_wait:;
	test	rdstatus,xrdybit;	ready ?
	jz	read_wait;		loop until data read

	mov	wrstrobe,dh;		middle byte of address set to high order

read_wait2:;
	test	rdstatus,xrdybit;	ready ?
	jz	read_wait2;		loop until data read

	mov	wrstrobe,dl;		low order byte of address

	xchg	dx,di;			get result record address in dx
	call	set_pointer;		leave pointer at result buffer

	pop	cx;			restore caller's registers
	pop	ax;
	ret;

set_command	endp;




;
;	set omninet ram pointer
;
;	inputs:		dx = address in omninet ram to set pointer at
;
;	outputs:	none
;
set_pointer	proc;

	mov	wrlocounter,dl;		set low-order byte of address
	mov	wrhicounter,dh;		set high-order byte of address

	ret;

set_pointer	endp;





code	ends;

	end;
