;=======================================================================
;
;	This collection of routines forms part of a generalised floppy
;	disk formatter for the Colex 850 computer.
;
;=======================================================================

;****************************************************************
;*								*
;*		EQUates for the Colex STD-FDC			*
;*								*
;****************************************************************

FDCBAS	EQU	0E0H		; Base address for the FDC
FDCDMA	EQU	FDCBAS		; Address of the z80 dma
FDCSEL	EQU	FDCBAS+3	; Address of drive/size/etc port
FDCCMD	EQU	FDCBAS+4	; Wd1797 command register
FDCSTA	EQU	FDCBAS+4	; Wd1797 status register
FDCTRK	EQU	FDCBAS+5	; Wd1797 track register
FDCSEC	EQU	FDCBAS+6	; Wd1797 sector register
FDCDAT	EQU	FDCBAS+7	; Wd1797 data register

MHZ	EQU	4		; Clock speed for timing loops

;-----------------------------------------------------------------------
;
;			FDC and DMA control bytes
;
;-----------------------------------------------------------------------

	PSECT	DATA

SELBITS:
	DEFB	00000001B	; Drive 0 select mask
	DEFB	00000010B	; Drive 1 select mask
	DEFB	00000100B	; Drive 2 select mask
	DEFB	00001000B	; Drive 3 select mask (8" only)
SELMSK:
	DEFB	0
FDOP:
	DEFB	0
DMABLOCK:
	DEFB	0C3H
	DEFB	79H
DMABUFPNT:
	DEFW	0		; Pointer to data buffer
	DEFW	10999		; No. of bytes/track
	DEFB	14H
	DEFB	28H
	DEFB	0C5H
	DEFB	FDCDAT
	DEFB	9AH
	DEFB	0CFH

READPBL	EQU	$-DMABLOCK	; Length of block for read

	DEFB	5
	DEFB	0CFH

WRTPBL	EQU	$-DMABLOCK	; Length of block for write

MOTOR	EQU	00001000B	; Motor on mask (5-1/4" only)
SDENS	EQU	10000000B	; Single density mask
FDCRES	EQU	01000000B	; Wd1797 reset
SIZE5	EQU	00100000B	; 5-1/4" size select mask
SIZE8	EQU	00000000B	; 8" size select mask
FRDSEC	EQU	10001000B	; Wd1797 sector read command
FWRSEC	EQU	10101000B	; Wd1797 sector write command
FSEEK	EQU	00011000B	; Wd1797 seek command
FRESTR	EQU	00001000B	; Wd1797 restore command
FRDID	EQU	11000100B	; Wd1797 read id field command
FRDTRK	EQU	11100000B	; Wd1797 read track command
FWRTRK	EQU	11110000B	; Wd1797 write track command

;-----------------------------------------------------------------------
;
;				M O U N T
;				=========
;
;	This routine does a hardware select and restore on the disk
;	drive named by the parameter passed on the stack.  In the
;	process it also builds the select mask for the drive for use
;	by other routines.
;
;	If the operation is successful then on exit the HL register
;	contains zero, otherwise it contains WD1797 error bits.
;
;	C calling sequence:
;
;		if (err=mount(drive,type,density,read_write))
;		  doError(err);
;		else
;		  . . . .	/* normal processing */
;
;-----------------------------------------------------------------------

	PSECT	TEXT
	GLOBAL	_MOUNT

_MOUNT:
	POP	AF		; Pull return address
	EX	AF,AF'          ;Park it out of the way for a moment
	POP	DE		; Get disk identifier,
	POP	BC		; Type
	POP	HL		; Density
	POP	AF		; Read/write mask
	PUSH	AF
	PUSH	HL
	PUSH	BC
	PUSH	DE
	EX	AF,AF'
	PUSH	AF		; Restore stack
	PUSH	HL		; Save density flag for a moment
	LD	HL,SELBITS	; Point at the select mask table
	ADD	HL,DE		; Address the mask for this drive
	LD	B,(HL)		; Pick up the drive select bits
	LD	A,C		; Get disk type
	OR	A		; 5-inch?
	JR	Z,1F		; Skip if not
	LD	A,MOTOR	.OR. SIZE5 ; Load motor-on mask and 5" size bit
1:
	OR	B		; Merge with drive select mask
	OR	FDCRES		; Suppress fdc reset
	POP	HL		; Recover density flag
	LD	H,A		; Park the select bits for a moment
	LD	A,L		; Move density flag to A for testing
	OR	A		; Test density flag
	LD	A,SDENS		; Assume it was single
	JR	NZ,1F		; Skip if correct
	XOR	A		; Otherwise clear single density flag
1:
	OR	H		; Add in the rest of the select bits
	LD	(SELMSK),A	; Save the select mask for other routines
	OUT	(FDCSEL),A	; Now select the drive

	LD	HL,0		; Delay for drive to come up to speed
1:
	IN	A,(FDCSTA)	; Get fdc status
	BIT	7,A		; Test if ready
	JR	Z,2F		; Done if so
	DEC	HL		; Adjust counter
	LD	A,H		; Get high byte
	OR	L		; Test with low byte
	JR	NZ,1B		; Loop if not done
2:
	LD	A,FRESTR	; Load restore command
	OUT	(FDCCMD),A	; Send command to floppy
	CALL	WAITFREE	; Wait for it to finish
	AND	11011000B	; Test for error conditions
	EX	AF,AF'          ;save test for a moment
	PUSH	AF		; Copy read/write mask
	POP	HL		; Into H
	EX	AF,AF'          ;restore test result
	AND	H		; Mask off write-protect bit if required
;
; Return result to calling program
;
	LD	L,A
	LD	H,0
	RET

;-----------------------------------------------------------------------
;
;			      D I S M O U N T
;			      ===============
;
;	De-selects all drives.	Called after a format or read track
;	operation so that the heads are unloaded and, for 5" drives,
;	the motor is turned off.
;
;-----------------------------------------------------------------------

	GLOBAL	_DISMOUNT
    
_DISMOUNT:
	LD	A,FDCRES	; Inhibit reset, clear all drives
	OUT	(FDCSEL),A
	RET

;-----------------------------------------------------------------------
;
;			      W A I T F R E E
;			      ===============
;
;	Called after issuing a command to the WD1797, this routine
;	simply waits for the completion of the command.
;
;-----------------------------------------------------------------------

WAITFREE:
	LD	A,15		; Load delay count
1:
	DEC	A		; Give time for WD1797 status register to
	JR	NZ,1B		; Become stable
2:
	IN	A,(FDCSTA)	; Get status
	BIT	0,A		; Test busy bit
	JR	NZ,2B		; Loop until not busy
	PUSH	AF
	LD	A,0C3H
	OUT	(FDCDMA),A	; Reset dma in case
	POP	AF
	RET

;-----------------------------------------------------------------------
;
;			R E A D T R A C K
;			=================
;
;	Given the address of a track buffer and the disk side number,
;	this routine does a raw data read of an entire track.
; 
;-----------------------------------------------------------------------

	GLOBAL	_READTRACK
    
_READTRACK:
	LD	A,FRDTRK	; Set up command for later
	JR	RWCOMMON	; Skip to common code

;-----------------------------------------------------------------------
;
;			W R I T E T R A C K
;			-------------------
;
;	Given the address of a pre-formatted track buffer and the disk
;	side number, this routine issues the commands to write-format
;	a track on the disk.
;
;-----------------------------------------------------------------------
    
	GLOBAL	_WRITETRACK
    
_WRITETRACK:
	LD	A,FWRTRK	; Set up command
;;	JR	RWCOMMON	; Drop through to common code

;-----------------------------------------------------------------------
;
;	Code common to read track and write track operations
;
;-----------------------------------------------------------------------

RWCOMMON:
	LD	(FDOP),A	; Save command
	EX	AF,AF'          ;Keep a copy
	POP	DE		; Return address
	POP	HL		; Track buffer address
	POP	BC		; Side number
	PUSH	BC		; Restore
	PUSH	HL		; Stack
	PUSH	DE		; For caller
	LD	(DMABUFPNT),HL	; Store buffer address in DMA program block
	LD	A,C		; Get side number
	EX	AF,AF'          ;Save if for later use and get command back
	CP	FWRTRK		; Which one are we doing?
	LD	B,WRTPBL	; Load length of params for write operation
	JR	Z,ISSUE		; Skip if write, otherwise ...
	LD	B,READPBL	; Load length of string for read operation
ISSUE:
    	LD	C,FDCDMA	; Point to dma port
	LD	HL,DMABLOCK	; Point at DMA program string
	OTIR			; Send to dma
	CALL	WAITINDEX	; Wait for end of index pulse
	LD	A,(SELMSK)	; Get the select mask
	OUT	(FDCSEL),A	; Send to the controller
	IN	A,(FDCDAT)	; Clear rubbish from fdc
	EX	AF,AF'          ;Recover the side number
	ADD	A,A		; Move side bit into position
	LD	HL,FDOP		; Point at track read/write command
	OR	(HL)		; Add in track operation command bits
	OUT	(FDCCMD),A	; Send command to floppy
	LD	A,87H
	OUT	(C),A		; Enable dma
	CALL	WAITFREE	; Wait for the FDC to become available
	AND	11100100B	; Test for errors
	LD	L,A		; Put things in the right spot
	LD	H,0
	RET			; Exit with result in HL
WAITINDEX:
	IN	A,(FDCSTA)	; Get floppy status
	AND	00000010B	; Check if during index
	JR	NZ,WAITINDEX	; Wait if so
	RET

;-----------------------------------------------------------------------
;
;				S T E P I N
;				===========
;
;	Steps the read/write heads one track towards the disk hub.
;
;-----------------------------------------------------------------------

	GLOBAL	_STEPIN

_STEPIN:
	LD	A,01011000B	; Load step-in command
	OUT	(FDCCMD),A	; Send command to floppy
	CALL	WAITFREE	; Wait for it to finish
	AND	11011000B	; Test for error conditions
	LD	L,A		; Return result to C program
	LD	H,0
	RET

;-----------------------------------------------------------------------
;
;			    S E C T O R S I Z E
;			    ===================
;
;	Builds a sector size code in the range 0 to 3 from an actual
;	sector size.
;
;	Sector size		size code
;
;	   128			    0
;	   256			    1
;	   512			    2
;	  1024			    3
;	  2048			    4
;
;-----------------------------------------------------------------------

	GLOBAL	_SECTORSIZE
    
_SECTORSIZE:
	POP	DE		; Return address
	POP	HL		; Parameter
	PUSH	HL
	PUSH	DE
	DEC	HL

; Sector size - 1 is in HL.  We want to shift right 7 bits but it is
; easier to shift left 1 bit and just use the H register contents.

	ADD	HL,HL
	LD	A,H
	LD	L,0		; Build result in HL
	LD	H,L
1:
    	BIT	0,A		; Test bit
	RET	Z		; Exit if reset
	SRL	A		; Otherwise shift down
	INC	L		; Update sector size code
	JR	1B		; Loop

;-----------------------------------------------------------------------
;
;				D E L A Y
;				=========
;
;	A simple delay routine for freezing the program for a specified
;	number of seconds.
;
;-----------------------------------------------------------------------
    
	GLOBAL	_DELAY
    
_DELAY:
	POP	DE		; Lift return address
	POP	HL		; Get number of seconds
	PUSH	HL		; Restore stack
	PUSH	DE
	PUSH	HL
	ADD	HL,HL		; *2
	ADD	HL,HL		; *4
	POP	DE
	ADD	HL,DE		; *5
	ADD	HL,DE		; *10
	INC	HL		; Pre-increment counter for test
1:
	DEC	HL		; Count down the deci-seconds
	LD	A,H		; Check for zero
	OR	L
	RET	Z		; Exit if time elapsed
	CALL	DELAY		; Wait .1 sec
	JR	1B
;
; .1 sec delay routine
;
DELAY:	LD	BC,4167*MHZ	; Timing constant * clock MHz

DELAY1:	DEC	BC
	LD	A,B
	OR	C
	JP	NZ,DELAY1
	RET

	END
