/*=============================================================================

				   M F 3 . C
				   =========

		  Floppy-disk formatting routines used by MF.C

=============================================================================*/

#include "stdio.h"
#include "mf.h"

extern byte *malloc();
extern char getKey();
extern void showError();
extern word mount();
extern word writeTrack();
extern void dismount();
char selectOne();

/*-----------------------------------------------------------------------------

				f o r m a t
				===========

	This routine formats a disk.

-----------------------------------------------------------------------------*/

format(drive,ft)
  struct Fmt *ft;
{
  short int tracks, trk;
  byte	goAhead;
  byte	*track;
  byte	fdError;
  char  *ps;
  static char perSide[] = " per side";

  tracks = ft->tpd;
  ps = &perSide[0];
  if (ft->dsMethod == ONE_SIDED || ft->dsMethod == LONG_TRACK)
    ps = &perSide[9];
  else
    tracks /= 2;

  goAhead = TRUE;

/* Make sure the disk is loaded in the drive.  Allow the option of quitting
   if the disk is not loaded. */

  do
    if (fdError=mount(drive-loDrive,ft->type,ft->density,WRITE))
      {
	showError(fdError);
	printf("\nPress ESC to abort, any other when disk is ready: ");
	if(getKey()==ESC)
	  goAhead = FALSE;
      }
  while (goAhead && fdError);

/* Now format the disk. */

  if (goAhead)
    {
      track = malloc(TRKBUFSZ);
      cls();
      highlight();
      printf("\n\n\nFormatting disk in drive %c as %s\n\n",
		drive, ft->name);
      printf("\t%d tracks of %d %d-byte sectors%s\n\n\n",
		tracks, ft->spt, ft->pss, ps);
      standard();
      cursor(OFF);
      for (trk=0;trk<tracks;++trk)
	{
	  layout(track,trk,0,ft);
	  if (fdError=writeTrack(track,0))
	    {
	      showError(fdError);
	      goto quit;
	    }
	  if (ft->dsMethod)		/* then must be 2-sided */
	    {
	      layout(track,trk,1,ft);
	      if (fdError=writeTrack(track,1))
		{
		  showError(fdError);
		  goto quit;
		}
	    }
	  if (trk < tracks-1)
	    if (fdError=stepIn())
	      {
		showError(fdError);
		goto quit;
	      }
	}
quit:
      cursor(ON);
      free(track);
      dismount();
      putchar('\n');
    }      
}

layout(track,trackNumber,side,ft)
  register byte *track;
  byte  trackNumber, side;
  struct Fmt *ft;
{
  register word count;
  byte sector, startSec, endSec;
  byte *tSave;
  byte ssc;
  static byte *secId[60];

/* Tell the human what is happening */

  printf("  Track %d side %d\r",trackNumber,side);

/* Build a sector size code. */

  ssc = sectorSize(ft->pss);

/* Record the starting address */

  tSave = track;

  if (trackNumber < 2)
    {
      /* Pre-index gap */

      for (count=ft->pigap;count--;)
	*track++ = ft->density==MFM ? 0x4E : 0xFF;

      for (count = (ft->density==FM ? 6 : 12); count--;)
	*track++ = 0;

      if (ft->density == MFM)
	for (count=3; count--;)
	  *track++ = 0xF6;	/* Write C2 with clock=E7 */

      /* Index mark */

      *track++ = INDEX_MARK;

      /* Gap 1 */

      for (count=ft->gap1;count--;)
	*track++ = ft->density==MFM ? 0x4E : 0xFF;
    }

/* Determine the sector count limits */

  startSec = 0;
  endSec = ft->spt;

  if (ft->dsMethod == LONG_TRACK)
    if (side)
      startSec = ft->spt / 2;
    else
      endSec   = ft->spt / 2;

/* Lay out the track data for each sector */

  for (sector=startSec; sector < endSec; ++sector)
    {
      if (trackNumber < 2)
	{
	  for (count=ft->gap3b;count--;)
	    *track++ = 0;
	  for (count=ft->gap3c;count--;)
	    *track++ = 0xF5;
	  secId[sector] = track;
	}
      else
	track = secId[sector];

/* Sector ID fields */

      *track++ = IDAM;			/* ID address mark */
      *track++ = trackNumber;
      *track++ = side;
      *track++ = ft->pSkew[sector];
      *track++ = ssc;
      *track++ = 0xF7;			/* Generates CRC */

/* Gap 2 */

      if (trackNumber < 2)
	{
	  for (count=ft->gap2a;count--;)
	    *track++ = ft->density==MFM ? 0x4E : 0xFF;
	  for (count=ft->gap2b;count--;)
	    *track++ = 0;
	  for (count=ft->gap2c;count--;)
	    *track++ = 0xF5;		/* Writes 0xA1 with special clock */

/* Data record (filled with 0xE5 or 0xF6) */

	  *track++ = DAM;		/* Data address mark */
	  for (count=ft->pss;count--;)
	    *track++ = 0xE5;
	  *track++ = 0xF7;		/* Generates CRC */

/* Gap 3 */

	  for (count=ft->gap3a;count--;)
	    *track++ = ft->density==MFM ? 0x4E : 0xFF;
	}
    } /* of things written for every sector */

/* Gap 4 */
    if (trackNumber < 2)
      {
	if ((count=TRKBUFSZ-(track-tSave)) > 1000)
          count = 1000;
	for(;count--;)
	  *track++ = ft->density==MFM ? 0x4E : 0xFF;
      }
}

/*-----------------------------------------------------------------------------

			    s e l e c t F o r m a t
			    =======================

	This routine displays a list of formats compatible with the disk
	drive selected and prompts for a selection.

-----------------------------------------------------------------------------*/

#define NO_SELECTION	-1
#define NEXT_SCREEN	-2
char selector[] = {"ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"};

struct Fmt *selectFormat(drive)
  char drive;
{
  int	formats, f, screen, screens;
  extern struct Fmt fmtSpec[];
  struct Fmt **fp, *spec, *ft[100];

  formats = 0;
  spec = &fmtSpec[0];
  fp = &ft[0];

  while (strlen(spec->name) && formats < 100)
    {
      if (compatible(spec,drive))
	{
	  *fp++ = spec;
	  ++formats;
	}
      ++spec;
    }

  f = NO_SELECTION;
  screens = (formats+35) / 36;
  screen = 0;
  if (screens > 0)
    while ((f=selectOne(ft,screen,drive,formats)) == NEXT_SCREEN)
      screen = (screen+1) % screens;
  if (f == NO_SELECTION)
    return (struct Fmt *)NULL;
  else
    return ft[f];
}

/*-----------------------------------------------------------------------------

			      c o m p a t i b l e
			      ===================

	Checks that a format is compatible with the selected disk drive.

-----------------------------------------------------------------------------*/

int compatible(format,drive)
   struct Fmt *format;
   char drive;
{
   extern char dkTypes[];
   return (format->type == dkTypes[drive-loDrive]);
}

/*-----------------------------------------------------------------------------

			       s e l e c t O n e
			       =================

	Displays a screenful of formats and prompts for a selection.
	Normally, user will enter a letter or digit to select a particular
	format but pressing the space bar progresses to the next screenful
	or pressing the return key aborts the selection process altogether.

-----------------------------------------------------------------------------*/

char *typeDescriptor[] = {"8-inch","5.25 inch 80-track","5.25 inch 40-track"};

char selectOne(ft,screen,drive,max)
  struct Fmt *ft[];
  int screen, max;
  char drive;
{
  int	b, e, f;
  char  s;
  extern char dkTypes[];

/* Clear the screen and display the heading */

  cls();
  highlight();
  printf("%d format%s compatible with %s floppy disk drive %c - Screen %d",
	max, max==1?"":"s",
	typeDescriptor[dkTypes[drive-loDrive]],
	drive, screen+1);
  standard();

/* Display the format descriptions for this screen */

  b = screen*36;
  if ((e = b+36) > max)
    e = max;

  for (f=b; f<e; ++f)
    {
      cursor(f%18 + 2, (f/18)*40 + 1);
      printf("%c  %s",selector[f],ft[f]->name);
    }

/* Display the prompt */

  cursor(23,0);
  highlight();
  printf("Press key to select format, or <cr> for none, or <sp> for more: ");
  standard();

/* Get a keystroke.  Allow any of the displayed selection characters or
   <return> or <space> */

  f = NEXT_SCREEN;
  while (f == NEXT_SCREEN)
    switch (s = getKey() & 0x5F)
      {  
case '\r':
	f = NO_SELECTION;
	break;
case ' ':
case '\0':
	f = NEXT_SCREEN;
	break;
default:
	for (f=0; f < e && s != selector[f]; ++f);
          if (f >= e)
	    f = NEXT_SCREEN;
      }
  putchar('\n');

/* Return the result to the caller */

  if (f == NO_SELECTION)
    return f;
  return b+f;
}
