/*
UNC.C - Uncrunch module of CP/M File eXpress
Copyright 20 Jan 1992 by 
Carson Wilson
1359 W. Greenleaf, #1D
Chicago, IL 60626

UUCP:	carson@sputnik.uucp
	..!uunet!ddsw1!carson

BBS:	Antelope Freeway, 1-708-455-0120

Notes:
	Adapted from UNCRunch/C by Frank Prindle.
*/

#include "cfx.h"

#define DUMBLINKER
#define	TABLE_SIZE  4096	/* size of main LZW table for 12 bit codes */
#define	XLATBL_SIZE 5003	/* size of physical translation table  */

/* Special values for predecessor in table */
#define	NOPRED 0x6fff		/* no predecessor in table */
#define	EMPTY  0x8000		/* empty table entry (xlatbl only) */
#define	REFERENCED 0x2000	/* table entry referenced if this bit set */
#define	IMPRED 0x7fff		/* impossible predecessor */

#define	EOFCOD 0x100		/* special code for end-of-file */
#define	RSTCOD 0x101		/* special code for adaptive reset */
#define	NULCOD 0x102		/* special filler code */
#define	SPRCOD 0x103		/* spare special code */

#ifdef        DUMBLINKER

/* Main LZW table and its structure */
struct entry
{	short predecessor;		/* index to previous entry, if any */
	unsigned char suffix;		/* character suffixed to previous entries */
} *table;

/* auxilliary physical translation table */
/* translates hash to main table index */
short *xlatbl;

/*byte string stack used by decode */
unsigned char *stack;

#else

struct entry
{	short predecessor;        /*index to previous entry, if any */
        unsigned char suffix;        /*character suffixed to previous entries */
} table[TABLE_SIZE];

/* Auxilliary physical translation table */
/*  translates hash to main table index */
short xlatbl[XLATBL_SIZE];

/* Byte string stack used by decode */
unsigned char stack[TABLE_SIZE];

#endif

/* Other Global Variables */

unsigned char codlen;		/* Variable code length in bits (9-12)	*/
short trgmsk;			/* Mask for codes of current length	*/
unsigned char fulflg;		/* Full flag - set once main table is full */
short entry;			/* Next available main table entry	*/
long getbuf;			/* Buffer used by getcode		*/
short getbit;			/* Residual bit counter used by getcode	*/
unsigned char entflg;		/* Inhibit main loop from entering this code */
int finchar;			/* First character of last substring output */
int lastpr;			/* Last predecessor (in main loop)	*/

void initb2();
int okflag;			/* Flags abort from nested routines */

unsigned char *stacklmt;		/* crw */

/* ---------------------------------------------------------------- */
/*
	Uncrunch a single file.
*/
void uncrunch(uncrfd, uncrfn, uncrfs)
FILE *uncrfd;
char *uncrfn;
long uncrfs;
{
	char *p;
	char outfn[80];			/* space to build output file name   */
	int pred;			/* current predecessor (in main loop)*/
	unsigned char reflevel; 	/* ref rev level from input file     */
	unsigned char siglevel; 	/* sig rev level from input file     */
	unsigned char errdetect;	/* error detection flag from input file*/
	unsigned file_cksum;		/* checksum read from input file     */

	/* Extract and build output file name */

	cgetc(uncrfd);	/* skip signature */
	cgetc(uncrfd);
	for ( p = outfn; (*p = (cgetc(uncrfd) & 0x7F)) != '\0' ; p++)
	{
		if (*p == '[')		/* comment separator */
		{
			*p = ' ';
			*++p = '[';
		}
		else if (*p == 1)	/* DateStamper separator */
			*p = '\0';
	}

	outcrlf(1);
	printf("  --> %s", outfn);

	*(cisubstr(outfn, ".") + 4) = '\0'; /* truncate non-name portion */

	if (diskout)
	{
#ifdef UNIX
		unixfn(outfn);
		printf(" (%s)", outfn);
#endif
		/* Open output file */
		if ( 0 == (outfd = fopen( outfn, "wb")) )
		{	cerror("can't create %s!", outfn);
			return;
		}
        }
	else
		outcrlf(2);

	/* read the four info bytes */
	reflevel  = cgetc(uncrfd);		/* currently not used */
	siglevel  = cgetc(uncrfd);
	errdetect = cgetc(uncrfd);
	cgetc(uncrfd);	/* skip spare */

	/* Make sure we can uncrunch this format file */
	/* note: this program does not support CRUNCH 1.x format */
	if (siglevel < 0x20 || siglevel > 0x2f)
	{	cerror("need newer UNCR version.");
		return;
	}

#ifdef        DUMBLINKER
	/* Allocate storage now for the big tables (keeps load short) */
	table	= (struct entry *) malloc(TABLE_SIZE * sizeof(struct entry ));
	xlatbl	= (short *) malloc(XLATBL_SIZE * sizeof(short));
	stack	= (unsigned char *) malloc(TABLE_SIZE * sizeof(char));
	if (table == NULL || xlatbl == NULL || stack == NULL)
	{       outcrlf(1);
		cerror("not enough memory to uncrunch.");
                goto uncrexit;
        }
#endif
	/* Initialize variables for uncrunching a file */
	intram();
	stacklmt = stack + TABLE_SIZE;

	/* Set up atomic code definitions */
	initb2();

	/* =============== MAIN DECODING LOOP ============= */
	outreccount = 0;
	pred = NOPRED;
	okflag = 1;
	while ((okflag)	&&			/* flags file corrupt */
		(contin))			/* file skip command flag */
	{
		lastpr = pred;                 /* Remember last predecessor */

		/* Read and process one code */
		if ((pred = getcode(uncrfd)) == EOFCOD)	/* End-of-file code */
			break;				/* All LZW codes read*/
		else if (pred == RSTCOD) /* reset code */
		{	entry = 0;
			fulflg = 0;
			codlen = 9;
			trgmsk = 0x1ff;
			pred = NOPRED;
			entflg = 1;
			initb2();
		}
		else	/* A normal code (nulls already deleted) */
		{       /* Check for table full */
			if (fulflg != 2)
				/* Strategy if table not full */
				if (decode(pred) == 0)
					enterx(lastpr, finchar);
				else
					entflg = 0;
			else
			{	/* Strategy if table is full */
				decode(pred);
				entfil(lastpr, finchar); /* attempt reassign */
			}
		}
	}
	/* Verify checksum if required */
	if ((errdetect == 0) && (contin))
	{	file_cksum = cgetc(uncrfd);
		file_cksum |= cgetc(uncrfd) << 8;
		cksum &= 0xFFFF;
		if (file_cksum != cksum)
			cerror("checksum error: expected: %04xh actual: %04xh", file_cksum, cksum);
	}
	uncrexit:
#ifdef DUMBLINKER
	free(table);
	free(xlatbl);
	free(stack);
#endif
}
/* ---------------------------------------------------------------- */
/*
	Initialize variables for each file to be uncrunched.
*/
void intram()
{
	trgmsk = 0x1ff;		/* Nine bits	*/
	codlen = 9;		/*  "		*/
	fulflg = 0;		/* Table empty	*/
	entry = 0;		/*  "		*/
	getbit = 0;		/* Buffer emtpy	*/
	entflg = 1;		/* First code always atomic	*/
	repeat_flag = 0;	/* Repeat not active */
	getbuf = 0L;		/* crw */
}
/* ---------------------------------------------------------------- */
/*
	Initialize the LZW and physical translation tables.
*/
void initb2()
{
	register int i;

	/* First mark all entries of xlatbl as empty */
	for (i = 0; i < XLATBL_SIZE; i++)
		xlatbl[i] = EMPTY;

	/* Enter atomic and reserved codes into LZW table */
	for (i = 0; i < 0x100; i++)
		enterx(NOPRED, i);	/* First 256 atomic codes */
	for (i = 0; i < 4; i++)
		enterx(IMPRED, 0);	/* Reserved codes */
}
/* ---------------------------------------------------------------- */
/*
	Enter the next code into the LZW table.
*/
void enterx(pred, suff)
int pred;		/* Table index of predecessor	*/
int suff;		/* Suffix byte represented by this entry */
{
	register struct entry *ep;
	ep = &table[entry];	/* entry is a global */

	/* Update xlatbl to point to this entry */
	figure(pred, suff);

	/* Make the new entry */
	ep->predecessor = (short)pred;
	ep->suffix = (unsigned char)suff;
	entry++;

	/* If only one entry of the current code length remains, update to */
	/*  next code length because main loop is reading one code ahead */
	if (entry >= trgmsk)
		if (codlen < 12)
		{
			/* Table not full, just make length one more bit */
			codlen++;
			trgmsk = (trgmsk << 1) | 1;
		}
		else
			/* Table almost full (fulflg==0) or full (fulflg==1) */
			/*  just increment fulflg - when it gets to 2 we     */
			/*  will never be called again */
			fulflg++;
}
/* ---------------------------------------------------------------- */
/*
	Find an empty entry in xlatbl which hashes from this
	  predecessor/suffix combo, and store the index of the
	  next available LZW table entry in it.
*/
void figure(pred, suff)
int pred;
int suff;
{
	short *hash();
	/* auto int disp; */
	int disp;
	register short *p;
	p = hash(pred, suff, &disp);

	/* Follow secondary hash chain as necessary to find an empty slot */
	while (((*p) & 0xffff) != EMPTY)
	{
		p += disp;
		if (p < xlatbl || p > xlatbl + XLATBL_SIZE)
			p += XLATBL_SIZE;
	}

	/* Stuff next available index into this slot */
	*p = entry;
}
/* ---------------------------------------------------------------- */
/*
	Hash pred/suff into xlatbl pointer.
	Duplicates the hash algorithm used by CRUNCH 2.3.
*/
short *hash(pred, suff, disploc)
int pred;
int suff;
int *disploc;
{
	register int hashval;

	hashval = ((((pred >> 4) & 0xff) ^ suff) | ((pred & 0xf) << 8)) + 1;
	*disploc = hashval - XLATBL_SIZE;
	return (xlatbl + hashval);
}
/* ---------------------------------------------------------------- */
/*
	Return a code of length "codlen" bits from the input file
	  bit-stream.
*/
getcode(gcodefd)
FILE *gcodefd;
{
	register int hole;
	int code;

	/* Always get at least a byte */
	getbuf = (getbuf << codlen) | (((long) cgetc(gcodefd)) << (hole = codlen - getbit));
	getbit = 8 - hole;

	/* If is not enough to supply codlen bits, get another byte */
	if (getbit < 0)
	{
		getbuf |= ((long) cgetc(gcodefd)) << (hole - 8);
		getbit += 8;
	}

	if (feof(gcodefd))
	{	cerror("unexpected EOF on input file.");
		return EOFCOD;
	}

	/* Skip spare or null codes */
	if ((code = ((getbuf >> 8) & trgmsk)) == NULCOD || code == SPRCOD)
		return getcode(gcodefd);	/* Skip this code, get next */
	return code;
}
/* ---------------------------------------------------------------- */
/*
	Decode this code.
*/
decode(code)
short code;
{
	register unsigned char *stackp; 	/* Byte string stack pointer */
	register struct entry *ep;

	ep = &table[code];

	if (code >= entry)
	{
		/* The ugly exception, "WsWsW" */
		entflg = 1;
		enterx(lastpr, finchar);
	}

	/* Mark corresponding table entry as referenced */
	ep->predecessor |= REFERENCED;

	/* Walk back the LZW table starting with this code */
	stackp = stack;

	/* Prevent overrun with corrupt files <crw> */
	while (ep > &table[255])	/* i.e. code not atomic */
	{
		*stackp++ = ep->suffix;
		ep = &table[(ep->predecessor)&0xfff];
		if (stackp > stacklmt)
		{
			cerror("CRUNCHed file corrupt: aborting.");
			okflag = 0;
			return(entflg);
		}
	}

	/* Then emit all bytes corresponding to this code in forward order */
	send(finchar = (ep->suffix) & 0xff);	/*first byte*/

	while (stackp > stack)			/*the rest*/
		send(*--stackp);

	return(entflg);
}
/* ----------------------------------------------------------------- */
/*
	Attempt to reassign an existing code which has been defined,
	  but never referenced.
*/
void entfil(pred, suff)
int pred;		/* Table index of predecessor	*/
int suff;		/* Suffix byte represented by this entry */
{
	auto int disp;
	register struct entry *ep;
	short *hash();
	short *p;
	p = hash(pred, suff, &disp);

	/* Search the candidate codes (all those which hash from this new */
	/*   predecessor and suffix) for an unreferenced one	*/
	while (*p != (short)EMPTY)
	{
        	/* Candidate code */
		ep = &table[*p];
		if (((ep->predecessor) & REFERENCED) == 0)
		{
			/* Entry reassignable, so do it! */
			ep->predecessor = pred;
			ep->suffix = suff;
			/* discontinue search */
			break;
		}
		/* Candidate unsuitable - follow secondary hash chain */
		/*  and keep searching	*/
		p += disp;
		if (p < xlatbl || p > xlatbl + XLATBL_SIZE)
			p += XLATBL_SIZE;
	}
}
/* End of UNC.C */
